<?php

namespace App\Http\Controllers;

use App\Models\Pelanggan;
use App\Models\Grub;
use App\Models\Pembayaran;
use App\Models\Site;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class PelangganController extends Controller
{
    // TAMPILAN PER GRUB
    public function index(Request $request)
    {
        $status = $request->get('status');
        $q = $request->get('q');
        $sort = $request->get('sort'); // untuk sort tabel
    
        // Ambil grub beserta relasi pelanggan dan pembayaran
        $grubs = Grub::with(['pelanggans.pembayaran'])
            ->withCount('pelanggans')
            ->get();
    
        // Filter by search (nama grub)
        if ($q) {
            $grubs = $grubs->filter(function ($grub) use ($q) {
                return stripos($grub->nama_grub, $q) !== false;
            });
        }
    
        // Hitung total tagihan dan belum lunas
        foreach ($grubs as $grub) {
            $allPayments = $grub->pelanggans->flatMap->pembayaran;
            $grub->total_tagihan = $allPayments->sum('jumlah_tagihan');
            $grub->total_tagihan_belum_lunas = $allPayments->where('status', 'belum_lunas')->sum('jumlah_tagihan');
        }
    
        // Filter by status tagihan (di level grub)
        if ($status) {
            $grubs = $grubs->filter(function ($grub) use ($status) {
                if ($status == 'lunas') {
                    return $grub->total_tagihan_belum_lunas == 0;
                } else { // belum_lunas
                    return $grub->total_tagihan_belum_lunas > 0;
                }
            });
        }
    
        // Sort by total tagihan belum lunas jika ada permintaan
        if ($sort == 'belum_lunas_asc') {
            $grubs = $grubs->sortBy('total_tagihan_belum_lunas');
        } elseif ($sort == 'belum_lunas_desc') {
            $grubs = $grubs->sortByDesc('total_tagihan_belum_lunas');
        }
    
        // Convert ke collection indexed agar paginate lancar
        $grubs = $grubs->values();
    
        // Paginate manual
        $perPage = 10;
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $currentItems = $grubs->slice(($currentPage - 1) * $perPage, $perPage)->all();
        $paginatedGrubs = new LengthAwarePaginator($currentItems, $grubs->count(), $perPage, $currentPage, [
            'path' => LengthAwarePaginator::resolveCurrentPath(),
            'query' => $request->query(),
        ]);
    
        return view('pelanggan.index', ['grubs' => $paginatedGrubs]);
    }

    // DETAIL GRUB: list pelanggan pada grub tersebut
    public function showGrub($id)
    {
        $grub = Grub::with(['pelanggans.pembayaran'])->findOrFail($id);
        return view('grubs.show', compact('grub'));
    }

    // DETAIL PELANGGAN: semua tagihan per bulan
    public function show($id)
    {
        $pelanggan = Pelanggan::with('pembayaran')->findOrFail($id);
        return view('pelanggan.show', compact('pelanggan'));
    }

    // FORM TAMBAH PELANGGAN
    public function create()
    {
        $grubs = Grub::all();
        $penagihs = \App\Models\User::role('penagihan')->get();
        return view('pelanggan.create', compact('grubs', 'penagihs'));
    }

    // SIMPAN PELANGGAN BARU
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'alamat' => 'required',
            'no_hp' => 'required',
            'paket_wifi' => 'required',
            'tanggal_berlangganan' => 'required|date',
            'grub_id' => 'required|exists:grubs,id',
            'penagih_id' => 'required|exists:users,id',
        ]);

        $pelanggan = Pelanggan::create([
            'nama' => $request->nama,
            'alamat' => $request->alamat,
            'no_hp' => $request->no_hp,
            'paket_wifi' => $request->paket_wifi,
            'tanggal_berlangganan' => $request->tanggal_berlangganan,
            'grub_id' => $request->grub_id,
            'penagih_id' => $request->penagih_id,
        ]);

        // Tagihan pertama otomatis
        $periode = Carbon::parse($pelanggan->tanggal_berlangganan)->format('Y-m');
        Pembayaran::create([
            'pelanggan_id'   => $pelanggan->id,
            'periode'        => $periode,
            'jumlah_tagihan' => $pelanggan->paket_wifi,
            'status'         => 'belum_lunas'
        ]);

        return redirect()->route('pelanggan.index')->with('success', 'Pelanggan berhasil ditambahkan!');
    }

    // FORM EDIT
    public function edit($id)
    {
        $pelanggan = Pelanggan::findOrFail($id);
        $grubs = Grub::all();
        $penagihs = \App\Models\User::role('penagihan')->get();
        return view('pelanggan.edit', compact('pelanggan', 'grubs', 'penagihs'));
    }

    // UPDATE PELANGGAN
    public function update(Request $request, $id)
    {
        $request->validate([
            'nama' => 'required',
            'alamat' => 'required',
            'no_hp' => 'required',
            'paket_wifi' => 'required',
            'tanggal_berlangganan' => 'required|date',
            'grub_id' => 'required|exists:grubs,id',
            'penagih_id' => 'required|exists:users,id',
        ]);

        $pelanggan = Pelanggan::findOrFail($id);
        $pelanggan->update([
            'nama' => $request->nama,
            'alamat' => $request->alamat,
            'no_hp' => $request->no_hp,
            'paket_wifi' => $request->paket_wifi,
            'tanggal_berlangganan' => $request->tanggal_berlangganan,
            'grub_id' => $request->grub_id,
            'penagih_id' => $request->penagih_id,
        ]);

        return redirect()->route('grubs.show', $request->grub_id)->with('success', 'Pelanggan berhasil diupdate!');
    }

    // HAPUS PELANGGAN
    public function destroy($id)
    {
        $pelanggan = Pelanggan::findOrFail($id);
        $pelanggan->delete();

        return redirect()->route('pelanggan.index')->with('success', 'Pelanggan berhasil dihapus!');
    }

    // TAGIHAN DETAIL PER PELANGGAN
    public function tagihan($id)
    {
        $pelanggan = Pelanggan::with('pembayaran')->findOrFail($id);
        return view('pelanggan.tagihan', compact('pelanggan'));
    }

    // TAGIHAN BELUM LUNAS
    public function tagihanBelumLunas($id)
    {
        $pelanggan = Pelanggan::findOrFail($id);
        $tagihanBelumLunas = $pelanggan->pembayaran()->where('status', 'belum_lunas')->get();

        return view('pelanggan.tagihan_belum_lunas', compact('pelanggan', 'tagihanBelumLunas'));
    }

    // GENERATE TAGIHAN BULANAN (opsional, bisa dipanggil dari tombol di index)
    public function generateTagihanBulanan()
    {
        $periode = Carbon::now()->format('Y-m');
        $pelanggans = Pelanggan::all();
        foreach ($pelanggans as $pelanggan) {
            $exists = $pelanggan->pembayaran()->where('periode', $periode)->exists();
            if (!$exists) {
                Pembayaran::create([
                    'pelanggan_id' => $pelanggan->id,
                    'periode' => $periode,
                    'jumlah_tagihan' => $pelanggan->paket_wifi,
                    'status' => 'belum_lunas'
                ]);
            }
        }
        return redirect()->route('pelanggan.index')->with('success', 'Tagihan bulanan berhasil digenerate!');
    }

    public function bayarCicilan(Request $request, $id)
{
    $pembayaran = Pembayaran::findOrFail($id);

    $jumlah_bayar = intval($request->jumlah_bayar);
    if ($jumlah_bayar <= 0) {
        return back()->with('error', 'Nominal cicilan harus lebih dari 0');
    }

    // Tambah nominal ke jumlah_dibayar
    $pembayaran->jumlah_dibayar += $jumlah_bayar;

    // Update status lunas jika sudah cukup
    if ($pembayaran->jumlah_dibayar >= $pembayaran->jumlah_tagihan) {
        $pembayaran->status = 'lunas';
        $pembayaran->jumlah_dibayar = $pembayaran->jumlah_tagihan; // pastikan tidak lebih dari tagihan
    } else {
        $pembayaran->status = 'belum_lunas';
    }

    $pembayaran->save();

    return back()->with('success', 'Pembayaran cicilan berhasil!');
}

public function suspend($id)
{
    $pelanggan = Pelanggan::findOrFail($id);
    $pelanggan->status = 'suspend';
    $pelanggan->save();

    return back()->with('success', 'Pelanggan berhasil di-suspend.');
}

public function aktifkan($id)
{
    $pelanggan = Pelanggan::findOrFail($id);
    $pelanggan->status = 'aktif';
    $pelanggan->save();

    return back()->with('success', 'Pelanggan telah diaktifkan kembali.');
}

}